    module Iqd_module
! parameters and arrays for calculating equivalent current function
!   associated with field-aligned currents above ggjtop(nggjhgt)
      integer, parameter :: nmax = 6
      integer :: m,n,p,po2,num,k
      real :: PMOPMMO(nmax+1), R(0:nmax,0:nmax), SQ2,pi,sq4pi
      real :: a(0:nmax,0:nmax,0:nmax), mc(0:nmax,0:nmax,0:nmax), &
        md(0:nmax,0:nmax)
      contains
!----------------------------------------------------------------------------- 
      subroutine qcoef
! Calculates coefficients needed to calculate functions Q of Richmond
!  (1974), including functions with odd n-m, used for symmetric
!  high-latitude FAC.  These functions are applied to the top of the
!  region of resolved 3D currents, above which current is assumed to
!  flow only along dipolar field lines out to the apex, and then
!  radially to infinity (for symmetric FAC).
!     
      implicit none
!     
      real :: x
      pi = 4.*atan(1.)
      SQ2 = sqrt(2.e0)
      sq4pi = sqrt(4.*pi)
      R = 0.
      do m=0,nmax
        if (m.ne.0) PMOPMMO(m) = sqrt(1. + .5/float(m))
        do n=max0(1,m),nmax
          R(n,m) = sqrt(float(n*n-m*m))/sqrt(float(4*n*n-1))
        enddo
      enddo
      a = 0.
      a(0,0,0) = 1.
      do m=1,nmax
        a(m,m,0) = sqrt(float(2*m+1)/float(2*m))*a(m-1,m-1,0)
        n = m + 1
        if (n.gt.nmax) cycle
        a(n,m,0) = sqrt(float(2*m+3))*a(m-1,m-1,0)
        n = m + 2
        if (n.gt.nmax) cycle
        a(n,m,0) = (1. - R(n-1,m)**2 - R(n-2,m)**2)*a(n-2,m,0)/ &
          (R(n,m)*R(n-1,m))
        a(n,m,1) = ((1. - R(n-1,m)**2 - R(n-2,m)**2)*a(n-2,m,1) &
           - a(n-2,m,0))/(R(n,m)*R(n-1,m))
        n = m + 3
        if (n.gt.nmax) cycle
        a(n,m,0) = (1. - R(n-1,m)**2 - R(n-2,m)**2)*a(n-2,m,0)/ &
          (R(n,m)*R(n-1,m))
        a(n,m,1) = ((1. - R(n-1,m)**2 - R(n-2,m)**2)*a(n-2,m,1) &
           - a(n-2,m,0))/(R(n,m)*R(n-1,m))
        if (m+4.gt.nmax) cycle
        do n=m+4,nmax
          do p=0,n-m,2
            po2 = p/2
            x = 0.
            if (p.ge.2) x = a(n-2,m,po2-1)
            a(n,m,po2) = ((1.-R(n-1,m)**2-R(n-2,m)**2)*a(n-2,m,po2) - &
              x - R(n-2,m)*R(n-3,m)*a(n-4,m,po2))/(R(n,m)*R(n-1,m))
          enddo
        enddo
      enddo
! Now multiply a_nmp of notes by m and divide by (2n-m-p).
      do m=1,nmax
        do n=m,nmax
          do p=0,n-m,2
            po2 = p/2
            a(n,m,po2) = m*a(n,m,po2)/float(2*n-m-p)
! a contains a_nmp of notes multiplied by m and divided by (2n-m-p).
          enddo
        enddo
      enddo
      mc = 0.
! Case 1: m,n are both odd
      do m=1,nmax,2
        do n=m,nmax,2
          mc(n,m,0) = a(n,m,0)
          if (2.gt.2*n-m-1) cycle
          do p=2,2*n-m-1,2
            po2 = p/2
            mc(n,m,po2) = a(n,m,po2) + &
              mc(n,m,po2-1)*float(2*n-m-p+1)/float(2*n-m-p)
! mc contains c_nm of notes multiplied by m.
          enddo
        enddo
      enddo
! Case 2: m,n are both even
      md = 0.
      do m=2,nmax,2
        do n=m,nmax,2
          mc(n,m,0) = a(n,m,0)
          do p=0,n-m,2
            po2 = p/2
            num = 2*n-m-p
            x = a(n,m,po2)*(num-1)
            do k=1,n-m/2
              if (num.eq.2) exit
              num = num-2
              x = x*(num-1)/float(num)
            enddo
            md(n,m) = md(n,m) + x
! md contains d_nm of notes multiplied by m.
          enddo
          if (2.gt.2*n-m-2) cycle
          do p=2,2*n-m-2,2
            po2 = p/2
            mc(n,m,po2) = a(n,m,po2) + &
              mc(n,m,po2-1)*float(2*n-m-p+1)/float(2*n-m-p)
! mc contains c_nm of notes multiplied by m.
          enddo
        enddo
      enddo

! Test
!      do m=0,nmax
!        do n=m,nmax
!          write (6,'(a1,2i3,7e10.3)') 'a',n,m,(a(n,m,po2),po2=0,nmax)
!        enddo
!      enddo
!      do m=0,nmax
!        do n=m,nmax
!          write (6,'(a2,2i3,7e10.3)') 'mc',n,m,(mc(n,m,po2),po2=0,nmax)
!        enddo
!      enddo
!      do m=0,nmax
!        do n=m,nmax
!          write (6,'(a2,2i3,7e10.3)') 'md',n,m,md(n,m)
!        enddo
!      enddo

! Case 3: If n-m is odd use array a, which contains 
!   a_nmp of notes multiplied by m and divided by (2n-m-p).
!
      end subroutine qcoef

!----------------------------------------------------------------------------- 
      subroutine calc_Iqd
! Currents are mapped from the rho grid to the QD grid, and then to a
!   Gaussian geographic grid for calculating ground magnetic perturbations.
! An equivalent current function corresponding to 3D currents above 
!   ggjtop(nggjhgt) is calculated on the geographic grid for this height.
! Height-integrated horizontal current densities in each current layer
!   between ggjtop(k-1) and ggjtop(k) are calculated on the geographic
!   grid for use in subroutines calc_Bcoef and calc_B to calculate
!   magnetic perturbations at the ground and within the ionosphere.
! 
      use params_module, only: nlat_qd,nlat_qd_h,nhgt_fix,nmlon,nmlat_h,rtd,nhgt_fix_r,re, &
!         hgt_fix_r,hgt_fix,pi,r0,rho_s, &
         hgt_fix_r,hgt_fix,r0,rho_s, &
         nmlon_h,nggjlon,nggjlat,nggjhgt,ggjlon,ggjclat,ggjhgt,ggjtop,ktop,iw,fw,ls,fs, &
         f11gg,f12gg,f21gg,f22gg,f31oFgg,f32oFgg
      use qd_module, only: qd_grid,qd,wgt1,jl_qd,wgt3,jl3_qd,lat_qd_ed, &
         LI1qd,I1qd,LI3qd,I3qd,I2qd,Jf1qd,Jf2qd,Jrqd,hgt_qd_mp,hgt_qd_ed,lat_qd_mp,lon_qd_mp, &
         q1,q2,q3, Keast,Ksouth,Jrtopgg,etagg, &
! Quantities on next two lines are not needed for delB calculations,
!   but can be useful for plotting.
	 Jr,Jf1,Jf2,g13,g23,Jf1hor,Jf2hor,Jeej,Jfac_qd,Je2J_qd,Je2J_qA, &
	 Je2J_qA1,Je2J_qA2
      use fieldline_s_module, only: fieldline_s1,fline_s1,fieldline_s2,fline_s2 
      use fieldline_r_module, only: fieldline_r,fline_r 
!     
      implicit none
!     
!      integer :: i,j,k,l,ll,isn,nlatmax,ip,im,jj
      integer :: i,j,l,ll,isn,nlatmax,ip,im,jj
      integer :: iwm,iwp,lm,lp,lm_eta,lp_eta,klayer
      real :: fac,fac1,fac2,dlatq,dlonq,I3_eq
      real,dimension(nmlon,nlat_qd-1,nggjhgt) :: Kf1qd,Jrtopqd
      real,dimension(nmlon,nlat_qd,nggjhgt) :: Kf2qd
      real :: delhgt,Jrbotgg,Jrgg,ggjbot
! AM 2022-08-08 mapping of LI1 extend the second index limits of the LI1 array to facilitate cubic spline fitting when interpolating to get LI1qd.
!      real :: LI1(nmlon,nmlat_h+1,nhgt_fix,2)     ! for mapping from mod.apex to quasi dipole grid latitudinal integrated I1
      real :: LI1(nmlon,0:nmlat_h+2,nhgt_fix,2)     ! for mapping from mod.apex to quasi dipole grid latitudinal integrated I1
! 2022-7-26 ADR Extend the second index limits of the LI3 array to facilitate cubic spline fitting when interpolating to get LI3qd.
      real :: LI3(nmlon,0:nmlat_h+2,nhgt_fix_r,2)   ! for mapping from mod.apex to quasi dipole grid latitudinal integrated I3
! orginal     real :: LI3(nmlon,nmlat_h+1,nhgt_fix_r,2)   ! for mapping from mod.apex to quasi dipole grid latitudinal integrated I3
      real :: lamqd_from_apex_coord, &   ! function
              latqd_tmp,absf1,f1f2
      real :: wte,wtw,wtn,wts,Jf1gg,Jf2gg,Kf1gg,Kf2gg

! parameters and arrays for calculating equivalent current function
!   associated with field-aligned currents above ggjtop(nggjhgt)
!!      integer, parameter :: mmax = 6 ,nmax = 6
!!      real :: pnmost(0:nmax,0:mmax),fm(-mmax:mmax) ! ,dpnm(0:nmax,0:mmax) 
      real :: pnmost(0:nmax,0:nmax),fm(-nmax:nmax) 
!!      real :: PMOPMMO(mmax+1), R(0:nmax,0:mmax), SQ2,sq4pi
!!      integer :: n,m,ma
      integer :: ma
!!      real :: nnp1,eta(nmlon,nlat_qd-1) &
!!              ,QCNST1,QCNST2,X,XL
      real :: nnp1,eta(nmlon,nlat_qd-1) 
!!      real :: C2(nmax,-mmax:mmax)
      real :: C2(nmax,-nmax:nmax)
! q = Qnm of Richmond (1974) when m is non-negative.  That paper has a
!  sign error when m is negative, when Qnm should reverse sign.
!!      real :: q(nmax,-mmax:mmax)
      real :: q(nmax,-nmax:nmax)
      real :: CT,ST,STS,CP,SP,PM2
      real :: ltto2,csm,s2n,sl
      
      logical, parameter:: debug=.true.  ! am_2024.11
!	      
!      real,parameter :: hgt_max = 1.5e5  ! [m] for testing
!      integer:: kmax_r

! variables for test cases
!      real :: theta,phi,radius,sq15,sq3
!      sq15 = sqrt(15.)
!      sq3 = sqrt(3.)

! ADR note 2015/10/8: Since LI1,LI3 are only used temporarily, they could be
!  made arrays with the single index j, if do loops are rearranged.  LI1qd,
!  LI3qd could also be 1D arrays, if I2qd is calculated in multiple steps.
!      
! calculated integrated current I1 (cannot be removed; same grid as a1 and aa1)
      LI1 =0.
! isn=1 in S hemisphere; isn=2 in N hemisphere
      do isn=1,2
	do i=1,nmlon
	  do k=1,nhgt_fix ! maximum of points on fieldline
            LI1(i,1,k,isn) =  0.   ! polar values
            nlatmax = nmlat_h-k+1  ! each height level loses one lat. grid point
	    do j=2,nlatmax+1  ! loop over all s2 points plus equator
                LI1(i,j,k,isn) = LI1(i,j-1,k,isn) + fline_s1(i,j-1,isn)%I1(k)
	    end do 
! AM 2022-08-08 mapping of LI1 Extrapolate LI3 across pole (for j=0) using quadratic fit to values at j=1,2,3
            LI1(i,0,k,isn) = 3.*LI1(i,1,k,isn) - 3.*LI1(i,2,k,isn) + LI1(i,3,k,isn)
! AM 2022-08-08 mapping of LI1 Extend LI1 across equator (for j=nlatmax+2)
            LI1(i,nlatmax+2,k,isn) = LI1(i,nlatmax+1,k,isn) + fline_s1(i,nlatmax,3-isn)%I1(k)
	  end do 
	end do
      end do
      
      if(debug) then
        isn=1
        do k=1,2
         do j=1,2
          do i=1,nmlon
           write(101,'(a10,3(x,i4),1x,e15.8)') 'LI1 isn=1 ',k,j,i,LI1(i,j,k,isn)
	  end do 
	end do
       end do
      end if 
!
! mapping of I1 from rho to qd coordinate system
      do isn=1,2
	do i=1,nmlon
	  do k=1,nhgt_fix ! maximum of points on fieldline
	    do l=1,nlat_qd_h-1  ! loop over all qd latitude edge points
	        j = jl_qd(l,k)
! AM 2022-08-08 mapping of LI1 
!	        LI1qd(i,l,k,isn) = (1-wgt1(l,k))*LI1(i,j,k,isn)+ wgt1(l,k)*LI1(i,j+1,k,isn)
	        LI1qd(i,l,k,isn) = wgt1(l,k,1)*LI1(i,j-1,k,isn) + wgt1(l,k,2)*LI1(i,j,k,isn) + wgt1(l,k,3)*LI1(i,j+1,k,isn) + wgt1(l,k,4)*LI1(i,j+2,k,isn)
	        if(l.lt.3.and.i.eq.10) then
                 write(108,'(a10,4(x,i4),9(x,e15.8))') 'LI1qdloop isn=1 ',k,j,i,l,LI1qd(i,l,k,isn),wgt1(l,k,1),LI1(i,j-1,k,isn), wgt1(l,k,2),LI1(i,j,k,isn),wgt1(l,k,3),LI1(i,j+1,k,isn), wgt1(l,k,4),LI1(i,j+2,k,isn)
	        end if
	    end do 
	    ! set equator value which is the last point for both grids so L1qd = LI1(rho) (assumes we always have an rho point at equator,
	    !                                                                              and we set up our grid that way)
	    j = nmlat_h-k+1  
	    l=  nlat_qd_h 
	    LI1qd(i,l,k,isn) = LI1(i,j+1,k,isn) 
	    !
	    ! calculate I1qd at midpoints l which is the difference of LI1qd from the adjacent edges)
	    ! eq. 242
	    do l=1,nlat_qd_h-1  ! loop over all qd latitude midpoints
	        I1qd(i,l,k,isn) = LI1qd(i,l+1,k,isn)-LI1qd(i,l,k,isn)  ! difference between integrated values is I1qd
	    end do 
	  end do
	end do
      end do 
      
      if(debug) then
        isn=1
        do k=1,nhgt_fix
         do j=1,2
          do i=10,10   ! nmlon
           write(102,'(a10,3(x,i4),1x,e15.8)') 'LI1qd isn=1 ',k,j,i,I1qd(i,j,k,isn)
	  end do 
	end do
       end do
      end if 
    
!  For I3qd:
! calculated integrated current I3
      LI3 =0.
      do isn=1,2
	do i=1,nmlon
	  do k=1,nhgt_fix_r ! maximum height level
            LI3(i,1,k,isn) =  0.   ! polar values
            nlatmax = nmlat_h-k+1  ! each height level loses one lat. grid point
	    do j=2,nlatmax  ! loop over s2 points
                LI3(i,j,k,isn) = LI3(i,j-1,k,isn) + fline_r(i,j-1,isn)%I3(k)
		latqd_tmp = lamqd_from_apex_coord(fline_s2(i,j-1,isn)%mlat_m,hgt_fix_r(k))
		!if(i.eq.20.and.k.eq.2) write(33,'(i4,2(x,e15.8))') l,latqd_tmp,LI3(i,l,k,isn)
	    end do 
	    ! 
!
!
!
! 2015-10-15 ADR: This calculation could be done within the above do
!   loop over j, if the do-loop upper limit is changed to nlatmax+1.
            j= nlatmax+1  ! integrated up to equator
            LI3(i,j,k,isn) = LI3(i,j-1,k,isn) + fline_r(i,j-1,isn)%I3(k)


! 2022-7-26 ADR Extrapolate LI3 across pole (for j=0) using quadratic fit to values at j=1,2,3
            LI3(i,0,k,isn) = 3.*LI3(i,1,k,isn) - 3.*LI3(i,2,k,isn) + LI3(i,3,k,isn)
! 2022-7-26 ADR Extend LI3 across equator (for j=nlatmax+2)
            LI3(i,nlatmax+2,k,isn) = LI3(i,nlatmax+1,k,isn) + fline_r(i,nlatmax,3-isn)%I3(k)
!!
!
!
	  end do
	end do
      end do 
      
      if(debug) then
        isn=1
        do k=1,2
         do j=1,2
          do i=1,nmlon
           write(103,'(a10,3(x,i4),1x,e15.8)') 'LI3 isn=1 ',k,j,i,LI3(i,j,k,isn)
	  end do 
	end do
       end do
      end if 
      
!      ! eq. 63' on page 7 for equator value 
!      do isn=1,2
!         do i=1,nmlon
!          if(i == 1) then
!            im = nmlon ! use wrap around point as i-1 -> nmlon 
!          else
!            im = i-1
!          endif
!          do k=1,nhgt_fix ! maximum of points on fieldline
!             j = nmlat_h-k+1  ! each height level loses one lat. grid point
!	     I3_eq = fline_r(i,j,isn)%I3(k)+fline_s1(im,j,isn)%I1(k)-fline_s1(i,j,isn)%I1(k)+fline_s2(i,j-1,isn)%I2(k)
!	  end do
!	end do
!      end do 
!
! mapping of I3 from rho to qd coordinate system
      do isn=1,2
	do i=1,nmlon
	  do k=1,nhgt_fix_r ! maximum of points on fieldline
	    do l=1,nlat_qd_h-1  ! loop over all qd latitude edge points
	        j = jl3_qd(l,k)
! 2022-7-26 ADR Use cubic spline interpolation instead of linear interpolation
                LI3qd(i,l,k,isn) = wgt3(l,k,1)*LI3(i,j-1,k,isn) + wgt3(l,k,2)*LI3(i,j,k,isn) + wgt3(l,k,3)*LI3(i,j+1,k,isn) + wgt3(l,k,4)*LI3(i,j+2,k,isn)
! orginal	LI3qd(i,l,k,isn) = (1-wgt3(l,k))*LI3(i,j,k,isn)+ wgt3(l,k)*LI3(i,j+1,k,isn)
!                write(99,'(4(a2,x,i4),3(x,f15.3))') 'i=',i,'k=',k,'l=',l,'j=',j,LI3(i,j,k,isn),LI3(i,j+1,k,isn),LI3qd(i,l,k,isn)
	    end do 
	    ! set equator value which is the last point for both grids so L3qd = LI3(rho) (assumes we always have an rho point at equator,
	    !                                                                              and we set up our grid that way)
	    j = nmlat_h-k+1
	    l=  nlat_qd_h 
	    LI3qd(i,l,k,isn) = LI3(i,j+1,k,isn) 
	    !
	    ! calculate I3qd at midpoints l which is the difference of LI3qd from the adjacent edges in latitude)
	    ! (eq. 246)
	    do l=1,nlat_qd_h-1  ! loop over all qd latitude midpoints
	        I3qd(i,l,k,isn) = LI3qd(i,l+1,k,isn)-LI3qd(i,l,k,isn)  
	    end do 
	  end do
	end do
      end do 
      
      if(debug) then
        isn=1
        do k=1,nhgt_fix
         do j=1,2
          do i=10,10   ! nmlon
           write(104,'(a10,3(x,i4),1x,e15.8)') 'LI3qd isn=1 ',k,j,i,I3qd(i,j,k,isn)
	  end do 
	end do
       end do
      end if 
!
! calculate I2qd as the divergence of L1qd and L3qd; L1qd longitudes are those of S1 grid with lon_s1(1) >  lon_s2(1)
! I2q(i,l+0.5,k) = L1q(i-0.5,l+0.5,k) - L1q(i+0.5,l+0.5,k)+
!                  L3q(i,l+0.5,k-0.5) - L3q(i,l+0.5,k+0.5)
      do isn=1,2
       do i=1,nmlon
	 im = i-1
	 if(i.eq.1) im = nmlon
	 do k=1,nhgt_fix ! maximum of points
	   I2qd(i,1,k,isn) = 0.  ! pole value
	   do l=2,nlat_qd_h      ! loop over all qd latitude edge points l+0.5
	     I2qd(i,l,k,isn) = LI1qd(im,l,k,isn)-LI1qd(i,l,k,isn)+ &
		      LI3qd(i,l,k,isn)-LI3qd(i,l,k+1,isn)	      
! At this point I2qd is equatorward current.  Change sign in N hemisphere to
!  make it be northward current in both hemispheres.
             if (isn.eq.2) I2qd(i,l,k,isn) = -I2qd(i,l,k,isn)
	   end do
	 end do
       end do
      end do 
      !
      if(debug) then
        isn=1
        do k=1,nhgt_fix
         do j=1,2
          do i=10,10   ! nmlon
           write(105,'(a10,3(x,i4),1x,e15.8)') 'L2qd isn=1 ',k,j,i,I2qd(i,j,k,isn)
	  end do 
	end do
       end do
      end if 
!
! At different points! eq. 254-256
! Jf1(i-0.5,l,k) = 0.5*I1qd(i-0.5,l,k)/r(k)^1.5/(r(k+0.5)^0.5-r(k-0.5)^0.5)/dlamq
! Jf2(i,l+0.5,k) = I2qd(i,l+0.5,k)/r(k)/(r(k+0.5)-r(k-0.5))/dlonq/cos(latq(l+0.5))
! Jr(i,l,k-0.5)  = I3qd(i,l,k-0.5)/M3qd(i,l,k-0.5)
!
      dlatq =  abs(lat_qd_mp(2)-lat_qd_mp(1))  ! regular grid spacing
      dlonq =  abs(lon_qd_mp(2)-lon_qd_mp(1))  ! regular grid spacing
!
      do k=1,nhgt_fix 
        do i=1,nmlon
          l = 1
          do l=2,nlat_qd_h  ! loop over all qd latitude midpoints (1 to 80)
	      isn = 1
             Jf1qd(i,l-1,k,isn) = I1qd(i,l-1,k,isn)/q1(l-1,k)
             Jf2qd(i,l,k,isn)	= I2qd(i,l,k,isn)/q2(l,k)
             Jrqd(i,l-1,k,isn)  = I3qd(i,l-1,k,isn)/qd(i,l)%M3q(k)	
	     !   
	      isn = 2
             ll = nlat_qd-1-l	! north pole to equator;  160 to 81 (midpoints) where M3q is stored
             Jf1qd(i,l-1,k,isn) = I1qd(i,l-1,k,isn)/q1(l-1,k)
             Jf2qd(i,l,k,isn)	= I2qd(i,l,k,isn)/q2(l,k)
             Jrqd(i,l-1,k,isn)  = I3qd(i,l-1,k,isn)/qd(i,ll)%M3q(k)   
          end do ! l    
        end do ! i
! Set Jf2qd at poles to the average of values either side of the pole.
        do i=1,nmlon_h  ! nmlon_h = nmlon/2
          do isn=1,2
            Jf2qd(i,1,k,isn) = .5*(Jf2qd(i,2,k,isn) - Jf2qd(i+nmlon_h,2,k,isn))
            Jf2qd(i+nmlon_h,1,k,isn) = -Jf2qd(i,1,k,isn) 
          enddo ! isn
        end do ! i
      end do ! k
      
      if(debug) then
        isn=1
        do k=1,nhgt_fix
         do j=1,2
          do i=10,10   ! nmlon
           write(106,'(a10,3(x,i4),3(x,e15.8))') 'Jf12rqd isn=1 ',k,j,i,Jf1qd(i,j,k,isn),Jf2qd(i,j,k,isn),Jrqd(i,j,k,isn)
           write(107,'(a10,3(x,i4),2(x,e15.8))') 'Jf12rqd isn=1 ',k,j,i,q1(j,k),qd(i,j)%M3q(k)
	  end do 
	end do
       end do
      end if 
!      	
      k=nhgt_fix_r 
        do i=1,nmlon
          do l=1,nlat_qd_h-1  ! loop over all qd latitude midpoints (1 to 80) in one hemisphere
	     isn = 1
             Jrqd(i,l,k,isn) = I3qd(i,l,k,isn)/qd(i,l)%M3q(k)  	   
	     isn = 2  
             ll = nlat_qd-l+1-1 ! north pole to equator;  160 to 81 (midpoints) where M3q is stored	   
             Jrqd(i,l,k,isn) = I3qd(i,l,k,isn)/qd(i,ll)%M3q(k)  	   
          end do 
        end do
! 
! Calculate height-integrated layer currents and radial current
!   densities at the top of each (thick) layer.
      Kf1qd = 0.
      Kf2qd = 0.
! ktop is the k index for the highest layer used for calculations of
!   currents, plus 1.  It is set in gen_ggj_grid.
!
      klayer = 1
      k = 0
      do while (klayer.le.nggjhgt) 
        k = k + 1
        if (hgt_fix_r(k).eq.ggjtop(klayer)) klayer = klayer + 1 ! increase thick layer index klayer if botteom of thin layer
	                                                        ! equals the top of the thick layer; then add current above
        if (klayer.gt.nggjhgt) exit 
          delhgt = hgt_fix_r(k+1) - hgt_fix_r(k)
          do l=1,nlat_qd_h-1  ! qd latitude midpoints or poleward edges (for
!                               Kf2qd) in S hemisphere
            ll = nlat_qd - l  ! N hemisphere latitude index for Kf1qd
            do i=1,nmlon
              Kf1qd(i,l ,klayer) = Kf1qd(i,l ,klayer) + delhgt*Jf1qd(i,l,k,1) 
              Kf1qd(i,ll,klayer) = Kf1qd(i,ll,klayer) + delhgt*Jf1qd(i,l,k,2) 
              Kf2qd(i,l ,klayer) = Kf2qd(i,l ,klayer) + delhgt*Jf2qd(i,l,k,1) 
              Kf2qd(i,ll+1,klayer)= Kf2qd(i,ll+1,klayer) + delhgt*Jf2qd(i,l,k,2)
! Note: the following step overwrites any previous Jrtopqd until klayer changes.
              Jrtopqd(i,l ,klayer) = Jrqd(i,l,k+1,1)
              Jrtopqd(i,ll,klayer) = Jrqd(i,l,k+1,2)
            enddo ! i
          enddo ! l
! Calculate Kf2qd at QD equator, averaging the S and N values, which
!   should be essentially the same.
          l = nlat_qd_h
          do i=1,nmlon
            Kf2qd(i,l,klayer) = Kf2qd(i,l,klayer) &
              + delhgt*(Jf2qd(i,l,k,1) + Jf2qd(i,l,k,2))*.5 
          enddo ! i
      enddo ! klayer.le.nggjhgt
! 
! Calculate equivalent current function eta corresponding to
!   field-aligned currents above ggjtop(nggjhgt) on QD grid.

!!      SQ2 = sqrt(2.e0)
!!      sq4pi = sqrt(4.*pi)
!!      do m = 0,mmax 
!!        if (m.ne.0) PMOPMMO(m) = sqrt(1. + .5/float(m))
!!        do n=m,nmax
!!          R(n,m) = sqrt(float(n*n-m*m)/amax1(4.*n*n-1.,1.))
!!        enddo
!!      enddo
!!      QCNST1 = 3.*sqrt(10.)/16.E0
!!      QCNST2 = 3.*sqrt(70.)/32.E0
      C2 = 0.
      do l=1,nlat_qd-1  ! loop over all qd latitude midpoints
! Select hemisphere and set isn and pole-to-equator index ll accordingly.
        isn = 1
        ll = l
        if (lat_qd_mp(l).gt.0.) then
          isn = 2
          ll = nlat_qd - l
        endif 
! Calculate q functions for this magnetic latitude.
        q = 0.
        CT = sin(lat_qd_mp(l))
        ST = cos(lat_qd_mp(l))
        STS = ST*ST               
!! Following formula for XL does not work at magnetic equator (CT = 0.),
!!  but lat_qd_mp does not include the equator.
!        XL = 1. + .5*STS*alog((1. + CT)/(1. - CT))/CT
!        q(1,1) = 1.2247449*CT*ST
!        q(2,2) = 1.3693064*CT*STS*XL
!        q(3,1) =  .2291288*CT*ST*(4. - STS*(3. + 6.*STS))
!        q(3,3) = 1.4790199*CT*STS*ST*(1. + 2.*STS)      
!        X = STS*(2. + 3.*STS*XL)                       
!        q(4,2) = QCNST1*CT*STS*(4. - X)             
!        q(4,4) = QCNST2*CT*STS*X                   
!        X = STS*(3. + STS*(4. + 8.*STS))            
!        q(5,1) = .018021728*CT*ST*(56. - STS*(188. - 13.*X))
!        q(5,3) =  .5255737*CT*STS*ST*(8. - X)              
!        q(5,5) =  .5484353*CT*STS*ST*X                    
!        X = STS*(8. + STS*(10. + 15.*STS*XL))            
!        q(6,2) = .057727979*CT*STS*(64. - STS*(168. - 3.*X))
!        q(6,4) = .079047290*CT*STS*STS*(80.  - 3.*X)       
!        q(6,6) =  .2140611*CT*STS*STS*X                   
        ltto2 = alog(tan((.5*pi-lat_qd_mp(l))/2.))
        csm = CT*ST
        do m=1,nmax
! First calculate q for even n-m 
          do n=m,nmax,2
            sl = 1.
            do p=0,2*n-m-1,2
              po2 = p/2
              q(n,m) = q(n,m) + csm*mc(n,m,po2)*sl
              sl = sl*STS
            enddo ! p
! If m and n are even, add the logarithmic term.
            if (mod(m,2).eq.0) q(n,m) = q(n,m) - md(n,m)*ltto2*STS**n
          enddo ! n
          csm = csm*ST
! Next calculate q for odd n-m 
          if (m+1.gt.nmax) cycle
          do n=m+1,nmax,2
            s2n = STS**n
            do p=0,n-m-1,2
              po2 = p/2
              q(n,m) = q(n,m) + a(n,m,po2)*(ST**(m+p)-s2n)
            enddo
          enddo ! n
        enddo ! m
        do n=1,nmax
!          do m=-mmax,-1
          do m=-nmax,-1
            q(n,m) = -q(n,-m)
          enddo ! m
        enddo ! n
        do i=1,nmlon
          CP = cos(lon_qd_mp(i))
          SP = sin(lon_qd_mp(i))
          fm(0) = 1.
          fm(-1) = SQ2*CP
          fm( 1) = SQ2*SP
!          do m=2,mmax
          do m=2,nmax
            fm(-m) = CP*fm(1-m) - SP*fm(m-1)
            fm( m) = CP*fm(m-1) + SP*fm(1-m)
          enddo ! m
          do n=1,nmax
            nnp1 = float(n*(n+1))
            fac = 1.e0/(nnp1*sq4pi)
!            do m=-mmax,mmax
            do m=-nmax,nmax
              C2(n,m) = C2(n,m) &
                - fac*I3qd(i,ll,ktop,isn)*q(n,m)*fm(-m)
            enddo ! m
          enddo ! n
        enddo ! i
      enddo ! l

      eta = 0.
      do l=1,nlat_qd-1
! CT,ST are cosine,sine of colatitude.
        CT = sin(lat_qd_mp(l))
        ST = cos(lat_qd_mp(l))
! pnmost = Pnm/ST, where Pnm is an associated Legendre polynomial,
!  fully normalized as in Richmond (1974).
        fac2 = ST/sq4pi
        pnmost = 0.
!        dpnm = 0.
        pnmost(0,0) = 1./ST
!        do m=0,mmax
        do m=0,nmax
          if (m.ne.0) pnmost(m,m) = PMOPMMO(m)*pnmost(m-1,m-1)*ST
!          dpnm(m,m) = m*CT*pnmost(m,m)
          PM2 = 0.
          do n=m+1,nmax
            pnmost(n,m) = (CT*pnmost(n-1,m) - R(n-1,m)*PM2)/R(n,m)
            PM2 = pnmost(n-1,m)
!            dpnm(n,m) = n*CT*pnmost(n,m) - float(2*n + 1)*R(n,m)*PM2
          enddo ! n
        enddo ! m
        do i=1,nmlon
          CP = cos(lon_qd_mp(i))
          SP = sin(lon_qd_mp(i))
          fm(0) = 1.
          fm(-1) = SQ2*CP
          fm( 1) = SQ2*SP
!          do m=2,mmax
          do m=2,nmax
            fm(-m) = CP*fm(1-m) - SP*fm(m-1)
            fm( m) = CP*fm(m-1) + SP*fm(1-m)
          enddo ! m
          do n=1,nmax
!            do m=-mmax,mmax
            do m=-nmax,nmax
              ma = iabs(m)
! Note: pnmost*fac2 is Pnm/sq4pi
              eta(i,l) = eta(i,l) + C2(n,m)*pnmost(n,ma)*fm(m)*fac2
            enddo ! m
          enddo ! n
        enddo ! i
      enddo ! l
! 
! Get currents on geographic grid for performing SH analysis.
      do i=1,nggjlon
        do j=1,nggjlat
          do k=1,nggjhgt
! Retrieve indices and weights for bilinear interpolation.
! Note: this would be easier if Jf1qd,Jf2qd,Jrqd had a single
!  latitude/hemisphere index from S to N, instead of l and isn.
! Each gg current point "x" falls in a QD lat-lon box; an example is
!  shown below with dotted lines.
! Points "O" are the locations of Jrqd; "|" are the locations of
!  Jf1qd; and "-" are the locations of Jf2qd.
! The western edge of the box lies on QD longitude lon_qd_mp(iw(i,j,k)),
!  with iw=1 corresponding to QD longitude = -pi.
! The southern edge of the box lies on QD latitude lat_qd_ed(ls(i,j,k)),
!  with ls=1 corresponding to QD latitude = -pi/2 (south QD pole).
! The fractional distance of the gg point from the western box edge
!  is fw(i,j,k).  In this example, fw>.5.
! The fractional distance of the gg point from the southern box edge
!  is fs(i,j,k).  In this example, fs>.5.
!
! lon_qd_: mp  ed  mp  ed  mp  ed    /lat_qd_: 
!
!           -       -       -               max=nlat_qd (N QD pole)
!
!           O   |   O   |   O   | lsp   mp  max=nlat_qd-1
!
!           -       - . . . -     lsp   ed
!                   .     x .
!           O   |   O   |   O   | ls    mp
!                   .       .
!           -       - . . . -     ls    ed
!
!           O   |   O   |   O   | lsm   mp  min=1
!
!           -       -       -     lsm   ed  min=1 (S QD pole)
!          iwm iwm iw  iw  iwp iwp   
!
!          min min         max max
!          =1  =1       =nmlon =nmlon   
!
!  The values of fw and fs correspond to the bilinear weights for
!  interpolating Jf2qd from the eastern and northern "-" points,
!  respectively.  The weights for the western and southern "-" points
!  are 1-fw, 1-fs, respectively.
!  To interpolate Jrqd ("O" points), the west and east indices and
!  weights are the same as for Jf2qd.  If fs>=.5 (as shown), the south
!  and north grid indices are the same as for Jf2qd but the south and
!  north weights change to 1.5-fs, fs-.5.  (However, if ls=nlat_qd-1
!  the south and north indices are reduced by 1, since there is no "O"
!  grid for nlat_qd, and the south and north weights become .5-fs,
!  .5+fs, respectively.)  If fs<.5 the south and north grid indices of
!  the points used for interpolation are decreased by 1 with respect to
!  those for Jf2qd (unless ls=1), and the south and north weights
!  change to .5-fs, .5+fs, respectively, unless ls=1, in which case the
!  south and north weights are 1.5-fs, fs-.5, respectively.
!  To interpolate Jf1qd ("|" points), the south and north indices are
!  the same as for interpolating Jrqd.  If fw>=.5 (as shown), the west
!  and east indices are the same as for Jrqd (and Jf2qd), but the west
!  and east weights change to 1.5-fw, fw-.5, respectively.  If fw<.5
!  the west and east indices shift 1 westward, and the west and east
!  weights become .5-fw, .5+fw.
!
! iwm = longitude index for western neighbor of iw, considering wraparound.
! iwp = longitude index for eastern neighbor of iw, considering wraparound.
            iwm = iw(i,j,k) - 1
            if (iwm.eq.0) iwm = nmlon
            iwp = iw(i,j,k) + 1
            if (iwp.gt.nmlon) iwp = 1
! First get indices and weights for interpolating Kf2qd, Jrqd, and eta
!  in longitude.
            im = iw(i,j,k)
            ip = iwp
            wte = fw(i,j,k)
            wtw = 1. - wte
! Next get indices and weights for interpolating Kf2qd in latitude.
            lm = ls(i,j,k)
            lp = lm + 1
            wtn = fs(i,j,k)
            wts = 1. - wtn
            Kf2gg        = wtw*(wts*Kf2qd(im,lm,k)  &
                              + wtn*Kf2qd(im,lp,k)) &  
                         + wte*(wts*Kf2qd(ip,lm,k)  &
                              + wtn*Kf2qd(ip,lp,k)) 
! Next get indices and weights for interpolating Jrqd, Kf1qd, and eta
!  in latitude.
            if (fs(i,j,k).lt..5) then 
              wtn = fs(i,j,k) + .5
              lm = ls(i,j,k) - 1
              lp = lm + 1
              if (lm.eq.0) then
! If gg point is within 1/2 QD latitude increment from QD south pole,
!  extrapolate Jf1qd, Jrqd, and eta in latitude toward pole.
                wtn = fs(i,j,k) - .5
                if(wtn.lt.0) wtn = 0
                lm = 1
                lp = 2
              endif
              if(lm.eq.0) then! test am 2022.08.26
                lm = lm + 2 
                lp = lm + 1 
              end if
              if(lm.eq.1) then! test am 2022.08.26
                lm = lm + 1 
                lp = lm + 1 
              end if
              if(lp.eq.nlat_qd-1) then! test am 2022.08.26
                lm = lm - 1 
                lp = lm + 1 
              end if
!!              lm_eta = lm
!!              lp_eta = lp
            else                          ! fs >= .5
              wtn = fs(i,j,k) - .5
              lm = ls(i,j,k) 
              lp = lm + 1 
              if(lp.eq.nlat_qd-1) then! test am 2022.08.26
                lm = lm - 1 
                lp = lm + 1 
              end if
              if(lm.eq.0) then! test am 2022.08.26
                lm = lm + 2 
                lp = lm + 1 
              end if
              if(lm.eq.1) then! test am 2022.08.26
                lm = lm + 1 
                lp = lm + 1 
              end if
!!              lm_eta = lm
!!              lp_eta = lp
! If gg point is within 1/2 QD latitude increment from QD north pole,
!  extrapolate Jf1qd, Jrqd, and eta in latitude toward pole.
              if (ls(i,j,k).eq.nlat_qd-1) then
                wtn = fs(i,j,k) + .5
                if(wtn.gt.1) wtn = 1.
!!                lm = 2
!!                lp = 1
!!                lp_eta = nlat_qd-1
!!                lm_eta = lp - 1
!                lp = nlat_qd-1
                lp = nlat_qd-2  ! test am 2022.08.26
                lm = lp - 1
              endif
            endif                 ! fs < .5 or >= .5
            
!            if(k.eq.14) then
!               write(44,'(2(i4,x),3(f15.7,x),2(i4,x))') i,j,fs(i,j,k),wtn,ls(i,j,k),lp,lm
!            end if
            
            wts = 1. - wtn
! Bilinear interpolation of etagg
            if (k.eq.nggjhgt) &
!!              etagg(i,j) = wtw*(wts*eta(im,lm_eta)  &
!!                                  + wtn*eta(im,lp_eta)) &  
!!                             + wte*(wts*eta(ip,lm_eta)  &
!!                                  + wtn*eta(ip,lp_eta)) 
              etagg(j,i) = wtw*(wts*eta(im,lm)  &
                              + wtn*eta(im,lp)) &  
                         + wte*(wts*eta(ip,lm)  &
                              + wtn*eta(ip,lp)) 
! Bilinear interpolation of Jrtopqd at top of layer (using indices
!  and weights for midpoint of layer as an approximation).
            Jrtopgg(j,i,k) = wtw*(wts*Jrtopqd(im,lm,k)  &
                                + wtn*Jrtopqd(im,lp,k)) &  
                           + wte*(wts*Jrtopqd(ip,lm,k)  &
                                + wtn*Jrtopqd(ip,lp,k)) 
! Next get indices and weights for interpolating Kf1qd in longitude.
            if (fw(i,j,k).lt..5) then
              wte = fw(i,j,k) + .5
              im = iwm
              ip = iw(i,j,k)
            else
              wte = fw(i,j,k) - .5
              im = iw(i,j,k)
              ip = iwp
            endif
            wtw = 1. - wte
! Bilinear interpolation of Kf1qd
            Kf1gg = wtw*(wts*Kf1qd(im,lm,k)  &
                       + wtn*Kf1qd(im,lp,k)) &  
                  + wte*(wts*Kf1qd(ip,lm,k)  &
                       + wtn*Kf1qd(ip,lp,k)) 
! Layer current densities in geographic eastward and northward directions;
!  first get contributions from Kf1 and Kf2.
            Keast(j,i,k)  =  Kf1gg*f11gg(i,j,k) + Kf2gg*f21gg(i,j,k)
            Ksouth(j,i,k) = -Kf1gg*f12gg(i,j,k) - Kf2gg*f22gg(i,j,k)
            if(k.eq.14.and.j.eq.165) then
               write(44,'(2(i4,x),7(e15.8,x),3(i4,x))') i,j,Keast(j,i,k),Ksouth(j,i,k),Kf1gg,Kf1qd(im,lm,k),Kf1qd(im,lp,k),Kf1qd(ip,lm,k),Kf1qd(ip,lp,k), &
                  im,ip,lp
            end if
          end do ! k
          Jrbotgg = 0.
          ggjbot = hgt_fix_r(1) 
          do k=1,nggjhgt
! Jr at midpoint heights
            Jrgg = .5*(Jrbotgg + Jrtopgg(j,i,k))
            delhgt = ggjtop(k) - ggjbot
! Add to Keast and Ksouth the horizontal components of Jf3 = F*Jr,
!  multiplied by layer thickness.
!  f31oFgg = f3(1)/F; f32oFgg = f3(2)/F
            Keast(j,i,k)  = Keast(j,i,k)  + Jrgg*f31oFgg(i,j,k)*delhgt
            Ksouth(j,i,k) = Ksouth(j,i,k) - Jrgg*f32oFgg(i,j,k)*delhgt 

! Test cases: overwrite Keast,Ksouth with simple model values
!            write(6,'(3i5,2e11.3)') j,i,k,Keast(j,i,k),Ksouth(j,i,k)
!            theta = ggjclat(j)
!            phi   = ggjlon(i)
!            radius= re + ggjhgt(k)
!          
!            !1
!            if(k.eq.1) then
!              Keast(j,i,k)  = sq15*cos(2.*theta)*cos(phi)/(re+110.e3)
!              Ksouth(j,i,k) = sq15*cos(theta)*sin(phi)/(re+110.e3)
!              Keast(j,i,k)  = sq15*cos(2.*theta)*cos(phi)/radius
!              Ksouth(j,i,k) = sq15*cos(theta)*sin(phi)/radius
!            else
!              Keast(j,i,k)  = 0.
!              Ksouth(j,i,k) = 0.
!            endif
!            Keast(j,i,k)  = sq15*cos(2.*theta)*cos(phi)/(re+110.e3)
!            Ksouth(j,i,k) = sq15*cos(theta)*sin(phi)/(re+110.e3)
!            
!            if(k.gt.1) then
!              Keast(j,i,k)  = 0.
!              Ksouth(j,i,k) = 0.
!            end if
!            etagg(j,i) = 0.
!            Jrtopgg(j,i,k) =0.
!            !2
!            Keast(j,i,k) = sq15*cos(2.*theta)*sin(phi)
!            Ksouth(j,i,k) = -sq15*cos(theta)*cos(phi)
!            !3
!            Keast(j,i,k) = -sq3*sin(theta)
!            Ksouth(j,i,k) = 0.
!           write(6,'(3i5,2e11.3)') j,i,k,Keast(j,i,k),Ksouth(j,i,k)
! Test cases: overwrite etagg with simple model values
!            etagg(j,i) = sq15*.5*sin(2.*theta)*cos(phi)
!            etagg(j,i) = sq15*.5*sin(2.*theta)*sin(phi)
!            etagg(j,i) = sq3*cos(theta)
!            etagg(j,i) = 0.

            Jrbotgg = Jrtopgg(j,i,k)
            ggjbot  = ggjtop(k)
          end do ! k
        enddo ! j
      enddo ! i
! 
! 
! Jf1,Jf2,Jr,Jf1hor,Jf2hor below are not used for computing delB
!   by the new procedure, but can be useful for plotting.
! 
! average to one i,l,k point   Jf1qd & Jrqd(i,l,k,isn) arrays are  Jf1qd_tmp,Jr,Jf2qd all at one point
      do k=1,nhgt_fix 
        do isn=1,2
          do i=1,nmlon
	    im = i-1
	    if(i.eq.1) im = nmlon
	    do l=1,nlat_qd_h-1  ! loop over all qd latitude midpoints
	       Jf1(i,l,k,isn) = 0.5*(Jf1qd(i,l,k,isn)+Jf1qd(im,l,k,isn) )   ! S1 points start later than S2 (-180) points in longitude
	       Jf2(i,l,k,isn) = 0.5*(Jf2qd(i,l,k,isn)+Jf2qd(i,l+1,k,isn) ) 
	       Jr(i,l,k,isn)  = 0.5*(Jrqd(i,l,k,isn)+ Jrqd(i,l,k+1,isn)) 
	   end do 
	 end do
       end do
      end do
      
! find index below hgt_max 
!      k=1
!      do while (hgt_max.ge.hgt_fix(k))
!         kmax_r = k
!	 k      = k+1
!      end do
!      	      
!      Jf1(:,:,kmax_r+1:nhgt_fix,:) = 0.
!      Jf2(:,:,kmax_r+1:nhgt_fix,:) = 0.
!      Jr(:,:,kmax_r+1:nhgt_fix,:)  = 0
!      .
!      Jf1(:,:,1:kmax_r,:) = 0.
!      Jf2(:,:,1:kmax_r,:) = 0.
!      Jr(:,:,1:kmax_r,:)  = 0.
! 
! calculate horizontal current J = Jf1hor*f1+Jf2hor*f2+Jr*k 
! Jf1hor = Jf1-(k^ dot g1)Jr (eq. 261)
! Jf2hor = Jf2-(k^ dot g2)Jr (eq. 262)
!
!
      do k=1,nhgt_fix 
        do isn=1,2
          do i=1,nmlon
	    do l=1,nlat_qd_h-1  ! loop over all qd latitude midpoints
	       Jf1hor(i,l,k,isn) = Jf1(i,l,k,isn) - g13(i,l,k,isn)*Jr(i,l,k,isn)
	       Jf2hor(i,l,k,isn) = Jf2(i,l,k,isn) - g23(i,l,k,isn)*Jr(i,l,k,isn)
	   end do 
	 end do
       end do
      end do 
!       
! calculate Jeej = (Jf1hor*f1+Jf1hor*f2)*f1/|f1|
!
      do k=1,nhgt_fix 
        do isn=1,2
          do i=1,nmlon
	    do l=1,nlat_qd_h-1  ! loop over all qd latitude midpoints
	       if(isn.eq.1) then  ! select hemisphere
		  jj = l
	       else
	          jj = nlat_qd-1-l+1 ! one hemisphere on qd midpoints goes from 1 to 80 ; pole to pole is 161
	       endif
	       absf1 = sqrt(qd(i,jj)%f11(k)**2+qd(i,jj)%f12(k)**2)
	       f1f2 = (qd(i,jj)%f11(k)*qd(i,jj)%f21(k)+qd(i,jj)%f12(k)*qd(i,jj)%f22(k))
	       Jeej(i,l,k,isn) = Jf1hor(i,l,k,isn)*absf1 + Jf2hor(i,l,k,isn)*f1f2/absf1 	           
	   end do 
	 end do
       end do
      end do    
!       
!
      do k=1,nhgt_fix
!        fac1 = (6.37122e6+hgt_qd_mp(k))/r0  !  r/R ; hgt_qd_mp [m]; 
! 6371.22 equatorial earth radius for aligned dipole
        fac1 = (re+hgt_qd_mp(k))/r0  !  r/R ; hgt_qd_mp [m]; 
        do isn=1,2
          do i=1,nmlon
	    do l=1,nlat_qd_h-1  ! loop over all qd latitude midpoints
	       if(isn.eq.1) then  ! select hemisphere
		  jj = l
	       else
	          jj = nlat_qd-1-l+1 ! one hemisphere on qd midpoints goes from 1 to 80 ; pole to pole is 161
	       endif
!	    
! calculate Jfac = Jf1_hor*f1*bo^+Jf1_hor*f2*bo^+ Jr*k*bo^
! F1 and F2 are horizontal since g3=Fk and f1=g2 x g3 and f2=g3 x g1
	       absf1 = qd(i,jj)%f11(k)*qd(i,jj)%bhat_mp(1,k)+ qd(i,jj)%f12(k)*qd(i,jj)%bhat_mp(2,k)
	       Jfac_qd(i,l,k,isn) = absf1* Jf1hor(i,l,k,isn)
	       absf1 = qd(i,jj)%f21(k)*qd(i,jj)%bhat_mp(1,k)+ qd(i,jj)%f22(k)*qd(i,jj)%bhat_mp(2,k)
	       Jfac_qd(i,l,k,isn) = Jfac_qd(i,l,k,isn) + absf1* Jf2hor(i,l,k,isn)
	       Jfac_qd(i,l,k,isn) = Jfac_qd(i,l,k,isn) + qd(i,jj)%bhat_mp(3,k)* Jr(i,l,k,isn)
!	
! calculate current Je2 Je2 = J*d2 = (Jf1hor*f1+Jf2hor*f2+Jr*k)*d2
	       Je2J_qd(i,l,k,isn) =qd(i,jj)%f1d2(k)*Jf1hor(i,l,k,isn) + &
	         qd(i,jj)%f2d2(k)*Jf2hor(i,l,k,isn)+ qd(i,jj)%kd2(k)*Jr(i,l,k,isn)

! alterative calculation: Je2 = -(R/r)cosI_m Jr - (r/R)^1.5*D*sinI*Jf2		 
	       Je2J_qA(i,l,k,isn) =-qd(i,jj)%cosIm(k)/fac1*Jr(i,l,k,isn)- fac1**1.5*qd(i,jj)%DsinI(k)*Jf2(i,l,k,isn)
	       Je2J_qA1(i,l,k,isn) =-qd(i,jj)%cosIm(k)/fac1*Jr(i,l,k,isn)
	       Je2J_qA2(i,l,k,isn) =- fac1**1.5*qd(i,jj)%DsinI(k)*Jf2(i,l,k,isn)
!	       
	   end do 
	 end do
       end do
      end do    
!         
      end subroutine calc_Iqd
!----------------------------------------------------------------------------- 
    end module Iqd_module
!----------------------------------------------------------------------------- 
