;+
; NAME:
;	ECI_TO_ECR
;
; PURPOSE:
;	Converts position vectors from ECI coordinates to ECR coordinates
;
; CATEGORY:
;	Utility
;
; CALLING SEQUENCE:
;	ECI_TO_ECR, year, doy, utc, eci, ecr
; 
; INPUTS:
;	year	Year, yyyy, longword integer
;	doy	Day of Year, ddd, longword integer
;	utc	Coordinated Universal Time of day in seconds, floating point
;       eci	Position vector in Earth-centered-intertial coordiantes, x, y, x, in km.
;
; OUTPUTS:
;	ecr	Position vector in Earthr-Centered-Rotational coordinates, x, y, z, in km
;
; KEYWORDS:
;	None
;
; COMMON BLOCKS:
;	None.
;
; PROCEDURE:
;	Transform position vectors in ECI (Earth-centered-inertial) coordinates
;	into ECR (Earth-centered-rotational) position vectors.
;	Uses SUNCOR to find Greenwich sidereal time (GST), the angle between
;	the Greenwich meridian and the vernal equinox.  
;	Arrays of vectors are OK (only 3 x N).
;	
; ROUTINES USED:
;	GST - calculates Greenwich sidereal time
;
; MODIFICATION HISTORY:
;       Stan Solomon, 3/15
;       Stan Solomon, 11/15, changed names from GEO to ECR
;
;-

pro eci_to_ecr, year, doy, utc, eci, ecr

; Get Greenwich sidereal time:
gst=gst(year,doy,utc)

; Calculate ECR position vector:
ecr=double(eci)
ecr[0,*]=eci[0,*]*cos(gst)+eci[1,*]*sin(gst)
ecr[1,*]=-eci[0,*]*sin(gst)+eci[1,*]*cos(gst)

return

end
