;+
; NAME:
;	ECR_TO_LLA
;
; PURPOSE:
;	Converts position vectors in ECR coordinates into Lat/Lon/Alt.
;
; CATEGORY:
;	Utility
;
; CALLING SEQUENCE:
;	lla = ecr_to_lla (ecr)
; 
; INPUTS:
;       ecr	ECR position vector, x, y, z, in km.
;
; OUTPUTS:
;	lla	latitude, longitude, altitude (degrees, degrees, km), double precision
;
; KEYWORDS:
;	None
;
; COMMON BLOCKS:
;	None.
;
; PROCEDURE:
;	Transform Earth-centered rotational (ECR) position vector into
;	geodetic latitude, longitude, and altitude above the surface.
;	Uses oblate spheroid approximation to shape of the Earth for altitude
;	and geodetic latitude calculation (ref.: W.J. Larson & J.R. Wertz,
;	Space Mission Analysis and Design, p. 809)
;	Arrays of vectors are OK (but no more than 3 x N)
;	
; MODIFICATION HISTORY:
;       Stan Solomon, 3/15
;       Stan Solomon, 11/15 - Changed names from GEO to ECR.
;
;-

function ecr_to_lla, ecr

; f = Earth oblateness flattening factor, re = equatorial radius:
f = 1./298.257D
re = 6378.14D

; Calculate length of position vector:
rs=sqrt(ecr[0,*]^2+ecr[1,*]^2+ecr[2,*]^2)

; Calculate normalized position vector:
rnx=ecr[0,*]/rs
rny=ecr[1,*]/rs
rnz=ecr[2,*]/rs

; Calculate declination, geodetic latitude, longitude, and altitude above oblate spheroid:
dec = asin(rnz)
lat = atan(tan(dec)/(1.-f)^2)
lon = atan(rny,rnx)
alt = re * (rs/re-(1-f)/(sqrt(1-f*(2-f)*(cos(dec))^2)))

; Convert radians into degrees:
lla=double(ecr)
lla[0,*] = lat * 180./!dpi
lla[1,*] = lon * 180./!dpi
lla[2,*] = alt

return,lla
end
