;+
; NAME:
;	LBHSPECFULL
;
; LICENSE:
;	This software is part of the GLOW model.  Use is governed by the Open Source
; 	Academic Research License Agreement contained in the file glowlicense.txt.
; 	For more information see the file glow.txt.
;
; PURPOSE:
;	Performs spectral synthesis of full LBH spectrum from GLOW output file
;
; CATEGORY:
;	Simulation
;
; CALLING SEQUENCE:  
;	Main program
;
; INPUTS:
;	GLOW netCDF output file containing global airglow and atmosphere fields for one time frame
;	Specified in first lines of program:
;		amodel		Atmosphere model identifier		string
;		gmodel		Airglow model identifier		string
;		nstart		number of first file to process		integer
;		nstop		number of last file to process		integer
;		nplots		number of image plots to make		integer
;               nwave           number of wavelength bins               integer
;		wavemin		minimum wavelength (Angstroms)		float
;
; OUTPUTS:  
;	LBH volume emission rate spectrum for full wavelength range
;
; COMMON BLOCKS:
;	None
;
; PROCEDURE:
;	Reads netCDF input file containing global airglow distribution in the far ultraviolet
;       calculated by the GLOW model, which used an empirical or numerical model as input.
;	Calculates LBH spectral distribution.
;
; ROUTINES CALLED:
;	LBHSPEC
;	I/O utilities:  READ_NETCDF, WRITE_NETCDF
;
; REFERENCES:
;	Solomon, S. C., "Observing system simulations for the GOLD mission,"
;	http://download.hao.ucar.edu/pub/stans/gold/pptx/GoldOssFeb2016.pptx
;
; MODIFICATION HISTORY:
;	v. 1.0     preliminary release for initial evaluation         Stan Solomon, 11/2017
;
;+

; --------------------------------------------------------------------

; Specify input and output file paths:

amodel='MSIS/IRI'
gmodel='GLOW v. 0.981'
directory='/hao/tethysv1/data/glowout/msisiri/marminlim/'
filecode='marmin'

infilepath=directory+'glow.'+filecode

; Specify file number to process:

ntime=10

; Set up wavelength grid:

nwave=3000                                 ; number of wavelength bins
apb=0.4                                    ; Angstroms per bin
wavemin=1200.2                             ; minimum wavelength in Angstroms
wavemax=wavemin+(nwave-1)*apb              ; maximum wavelength in Angstroms
wave=wavemin+findgen(nwave)*apb            ; array of bin center wavelengths in Angstroms
waves=wave-apb/2.                          ; array of bin starting boundary wavelengths

; Find bins corresponding to important wavelengths:

w1410=value_locate(waves,1410.)
w1528=value_locate(waves,1528.)
w1672=value_locate(waves,1672.)
w1812=value_locate(waves,1812.)

  ; Read netCDF file containing output from GLOW:

  infile=infilepath+'.'+string(ntime,format='(i3.3)')+'.nc'
  read_netcdf,infile,d

  ; Obtain model parameters and set up arrays:

  nlon=n_elements(d.lon)
  nlat=n_elements(d.lat)
  nlev=n_elements(d.lev)
  dlon=360./nlon
  dlat=180./nlat
  lonmin=-180.
  lonmax= 180-dlon
  latmin= -90+dlat
  latmax=  90-dlat
  date=d.idate
  ut=d.ut
  f107a=d.f107a
  f107=d.f107
  ap=d.ap
  lon=d.lon
  lat=d.lat
  zcm=d.zzz
  zkm=zcm/1.e5
  ao=d.ao
  ao2=d.ao2
  an2=d.an2
  atn=d.atn

  ; Estimate total LBH volume emission & column brightness 137-160, 141-153, (3,0) band at 135.4 nm:

  shortfac=0.23                               ; estimated ratio LBH 141-153 nm to total band system
  lbhsys=reform(d.eta[*,*,*,11])

  ; Calculate LBH synthetic spectrum using N2(a)v' distribution from GLOW output:

  spec=fltarr(nlon,nlat,nlev,nwave)
  for k=0,nlev-1 do begin
    for j=0,nlat-1 do begin
      for i=0,nlon-1 do begin
        t=atn[i,j,k]
        vplbh=reform(d.lbh[i,j,k,*])
        lbhspec,wavemin,wavemax,t,vplbh,wavelbh,spectrum
        spec[i,j,k,*]=spectrum
      endfor
    endfor
  endfor

  ; Calculate LBH wavelength integrals:

  lbhs=total(spec[*,*,*,w1410:w1528],4)
  lbhl=total(spec[*,*,*,w1672:w1812],4)
  lbht=total(spec,4)

end
