;+
; NAME:
;	LLA_TO_ECR
;
; PURPOSE:
;	Converts Lat/Lon/Alt coordiantes into ECR position vectors
;
; CATEGORY:
;	Utility
;
; CALLING SEQUENCE:
;	ecr = lla_to_ecr (lla)
; 
; INPUTS:
;	lla	latitude, longitude, altitude (degrees, degrees, km)
;
; OUTPUTS:
;       ecr	ECR position vector, x, y, z, in km.
;
; KEYWORDS:
;	None
;
; COMMON BLOCKS:
;	None.
;
; PROCEDURE:
;	Transform geodetic latitude, longitude, and altitude above the surface
;	into position vector in ECR (Earth-centered rotational) coordinates
;	Uses oblate spheroid approximation to shape of the Earth for altitude
;	and geodetic latitude calculation (ref.: W.J. Larson & J.R. Wertz,
;	Space Mission Analysis and Design, p. 809)
;	Arrays of vectors are OK (but no more than 3 x N)
;	
; MODIFICATION HISTORY:
;       Stan Solomon, 3/15
;       Stan Solomon, 11/15 - Changed names from GEO to ECR
;
;-

function lla_to_ecr, lla

; f = Earth oblateness flattening factor, re = equatorial radius:
f = 1./298.257D
re = 6378.14D

; Convert degrees into radians:
lat = lla[0,*] * !dpi/180.
lon = lla[1,*] * !dpi/180.
alt = lla[2,*]

; Calculate declination:
dec = atan(tan(lat)*(1.-f)^2)

; Calculate normalized position vector:
rnx = cos(lon)*cos(dec)
rny = sin(lon)*cos(dec)
rnz = sin(dec)

; Calculate length of position vector:
rs = alt + re * (1-f)/(sqrt(1-f*(2-f)*(cos(dec))^2))

; Calculate position vector:
v=double(lla)
v[0,*]=rnx*rs
v[1,*]=rny*rs
v[2,*]=rnz*rs

return, v

end
