;+
; NAME:
;	INTERGRID
;
; PURPOSE:
;	Performs grid interpolation using pre-calculated interpolation factors.
; 	Called by GOLDLOS and other line-of-sight calculation programs.
;
; CATEGORY:
;	Utility
;
; CALLING SEQUENCE:  
; INTERGRID,field,lonp1,lonp2,latp1,latp2,altp1,altp2,lonfac1,lonfac2,latfac1,latfac2,altfac,value
;
; INPUTS:
;	field:  3D array of quantities to be interpolated; float
;	lonp*:  3D array of precalculated indices of surrounding longitudes; integer
;	latp*:  3D array of precalculated indices of surrounding latitude; integer
;	altp*:  3D array of precalculated indices of surrounding altitudes; integer
;	*fac*:  3D array of precalculated interpolation factors for lon, lat, alt; float
;
; OUTPUTS:  
;	value:  3D array of interpolated values along lines of sight; float
;
; COMMON BLOCKS:
;	None
;
; PROCEDURE:
;	Performs bi-linear interpolation in longitude and latitude.
;	Performs logarithmic interpolation in altitude
;
; ROUTINES CALLED:
;	None
;
; MODIFICATION HISTORY:
;	Stan Solomon, 2015
;
;+
pro intergrid,field,lonp1,lonp2,latp1,latp2,altp1,altp2,lonfac1,lonfac2,latfac1,latfac2,altfac,value

y1 = field[lonp1,latp1,altp1]*lonfac1*latfac1 $
   + field[lonp1,latp2,altp1]*lonfac1*latfac2 $
   + field[lonp2,latp1,altp1]*lonfac2*latfac1 $
   + field[lonp2,latp2,altp1]*lonfac2*latfac2

y2 = field[lonp1,latp1,altp2]*lonfac1*latfac1 $
   + field[lonp1,latp2,altp2]*lonfac1*latfac2 $
   + field[lonp2,latp1,altp2]*lonfac2*latfac1 $
   + field[lonp2,latp2,altp2]*lonfac2*latfac2

value=y1*(y2/y1)^altfac

wbad=where(finite(value) eq 0)
if wbad[0] ne -1 then value[wbad]=0.

return

end
