;+
; NAME:
;	LBHSPEC
;
; PURPOSE:
;	Calculate LBH synthetic spectrum using lookup table
;
; CATEGORY:
;	Simulation
;
; CALLING SEQUENCE:  
;	lbhspec, wavemin, wavemax, t, vplbh, wave, spect
;
; INPUTS:
;	wavemin:  minimum wavelength, integer, A (must be between 1200 and 2399)
;       wavemax:  maximum wavelength, integer, A (must be between 1200 and 2399)
;       t:        temperature, K, real
;       vplbh:    array of lbh v'=0-9 population rates, cm-3 s-1, real, 10 elements
;
; OUTPUTS:  
;       wave:     wavelengths of spectrum, 0.4 A bins, each bin centered at midpoints, real
;	spect:    spectrum in 1-A bins, cm-3 s-1 per bin, real
;
; COMMON BLOCKS:
;	table:    holds table data read from file on first call
;
; PROCEDURE:
;	Reads LBH lookup table from file 'lbhtable.nc' finds spectrum at given temperature for
;       each upper state v'=0-9 of LBH bands, sums output spectrum for all bands.
;       (Note that v'=7-9 excitation rates are generally zero because they predissociate). 
;       Each v' spectra at each t in the table is normalized so that the sum of the bins = 1.0
;       User is expected to provide v' populations in appropriate ratios
;
; ROUTINES CALLED:
;	None.
;
; REFERENCES:
;	LBH model written by Scott Budzien
;
; MODIFICATION HISTORY:
;       Stan Solomon, 3/15, earlier version using output from Cleary model
;	Stan Solomon, 1/16, no changes except to lbhtable.nc, now based on Budzien model
;       Stan Solomon, 8/16, corrected comment on units of output spect
;       Stan Solomon, 2/17, changed from 1 A to 0.4 A resolution
;
;+

pro lbhspec, wavemin, wavemax, t, vplbh, wave, spect

common lbhtable, temp, vp, w, table

stemp=size(temp)
if stemp(0) eq 0 then begin
  read_netcdf,'lbhtable.nc',a
  temp=a.temperature
  vp=a.vprime
  w=a.wavelength
  table=a.lbhtable
endif

w1=fix((wavemin-1200.19)/0.4)
w2=fix((wavemax-1200.19)/0.4)
tf=fix(t)

wave=w[w1:w2]
spect = reform(table[w1:w2,*,tf]) # vplbh

return

end
