;+
; NAME:
; 	PLOTIMAGES
;
; PURPOSE:
;	Makes image plots of UV airglow output from GOLDLOS
;	Optionally saves images to plot files
;
; CATEGORY:
;	Graphics
;
; CALLING SEQUENCE:  
;	plotimages,ntime,plotfilepath,writeplots,nplots,date,ut,amodel,gmodel, $
;	satlat,satlon,satrad,scbo5s,scblbhs,scblbh30,rat,teff
;
; INPUTS:
;	Output file path, legends, and fields from GOLDLOS
;
; OUTPUTS:  
;	Plots to screen windows and optionally to files
;
; COMMON BLOCKS:
;	None
;
; PROCEDURE:
;	Makes global images of O 135.6 and LBH emission, their ratio, and effective temperature 
;
; ROUTINES CALLED:
;	Uses IDL MAP_SET utilities and superimposes a scaled image using TV
;
; MODIFICATION HISTORY:
;	Stan Solomon, 2016
;
;+


pro plotimagestemp,ntime,nplots,plotfilepath,date,ut,amodel,gmodel,satlat,satlon,satrad, $
               scbo5s,scblbhs,scblbh30,rat,teff,writeplots,nodisplay
               

    dm2=(!d.table_size-2)
    for plots=3,3 do begin
      window,0,xsize=900,ysize=900,pixmap=nodisplay
      map_set,0,satlon,satlat,/satellite,sat_p=[satrad,0,0],/nobo,xmargin=[9,25],ymargin=[10,10]
      if plots eq 1 then begin
        max=4 
        min=0
        b = dm2 * (alog10(scbo5s+scblbh30)-min) / (max-min) < dm2 > 0
        colorbry
        colorbar,min,max,(max-min),.92,.2,.015,.60
        xyouts,.92,.88,'1356',/normal,size=1.5
        xyouts,.92,.84,'(log R)',/normal,size=1.5
      endif
      if plots eq 2 then begin
        max=4 
        min=0
        b = dm2 * (alog10(scblbhs)-min) / (max-min) < dm2 > 0
        colorbry
        colorbar,min,max,(max-min),.92,.2,.015,.60
        xyouts,.92,.88,'LBHs',/normal,size=1.5
        xyouts,.92,.84,'(log R)',/normal,size=1.5
      endif
      if plots eq 3 then begin
        max=2
        min=0
        b = dm2 * ((rat-min)/(max-min)) < dm2 > 0
        colorbry
        colorbar,min,max,max-min,.92,.2,.015,.60
        xyouts,.92,.88,'OI/LBH',/normal,size=1.5
      endif
      if plots eq 4 then begin
        max=1000
        min=500
        b = dm2 * (teff-min)/(max-min) < dm2 > 0
        loadct,2
        colorbar,min,max,5,.92,.2,.015,.60
        xyouts,.92,.88,'T!deff!n',/normal,size=1.5
        xyouts,.92,.84,'(K)',/normal,size=1.5
      endif
      tv,rebin(b,800,800,/sample),0,50
      map_continents
      map_grid,latdel=10,londel=10
      xyouts,.03,.94,'Date '+strtrim(date,2)+'  UT '+strtrim(long(ut),2),size=1.5,/normal
      xyouts,.03,.03,amodel,/normal,size=1.2
      xyouts,.87,.03,gmodel,/normal,size=1.2

      ; output plots to files if requested:

      if writeplots then begin
        if plots eq 1 then plotfile=plotfilepath+'.O5S.'+string(ntime,format='(i3.3)')
        if plots eq 2 then plotfile=plotfilepath+'.LBH.'+string(ntime,format='(i3.3)')
        if plots eq 3 then plotfile=plotfilepath+'.RAT.'+string(ntime,format='(i3.3)')
        if plots eq 4 then plotfile=plotfilepath+'.TEF.'+string(ntime,format='(i3.3)')
        image=tvread(filename='diskimage',/nodialog,/gif)
      endif

    endfor

return

end

