;+
; NAME:
;	RTO5S
;
; PURPOSE:
;	Calculate total final source functions for the O(5S) doublet at 135.6 nm and 135.9 nm
;       using simplified monochromatic radiative transfer algorithm.
;
; CATEGORY:
;	Simulation
;
; CALLING SEQUENCE:  
;	rto5s,zcm,ox,o2,tn,vero5s,tfsf1356,tfsf1359 (,init,sssf,dssf,tssf)
;
; INPUTS:
;	zcm:      altitude, cm
;       ox:       atomic oxygen density, cm-3
;       o2:       molecular oxygen density, cm-3
;       tn:       temperature, K
;       vero5s:   volume excitation rate of the O(5S) upper state, cm-3 s-1
;
;       All inputs are assumed to be 3D arrays in (lon,lat,lev)
;       lon and lat are arbitrary, but lev must be the atmospheric levels in ascending order
;
; OUTPUTS:  
;       tsf1356:  total final source function of the 135.6 nm emission line, cm-3 s-1
;       tsf1359:  total final source function of the 135.9 nm emission line, cm-3 s-1
;       (optional outputs init, sssf, dssf, tssf for analysis of scattering source functions.)
;
;       All outputs are 3D arrays in (lon,lat,alt)
;
; COMMON BLOCKS:
;       none
;
; PROCEDURE:
;       Allocate excitation into 135.6 and 135.9 lines according to excitation cross sections.
;       Calculate 135.6 single- and double-scattered source functions throughout atmosphere,
;       considering resonant scattering by O and absorption by O2.
;       Allocate scattered photons to 135.6 and 135.9 total final source functions
;       according to transition coefficients.
;       135.9 scattering by O is considered to be negligible.
;       Assumes plane-parallel, 1D atmosphere.
;       Extensively calibrated using G.R. Gladstone's "REDISTER" Feautrier PFR code
;
; ROUTINES CALLED:
;	None.
;
; REFERENCES:
;       R.R. Meier, SSR, 58, 1, 1991; G.R. Gladstone, JGR, 93, 14623, 1988.
;
; MODIFICATION HISTORY:
;	Stan Solomon, 1/16
;
;+

pro rto5s,zcm,ox,o2,tn,vero5s,tfsf1356,tfsf1359,init,sssf,dssf,tssf

; Get array dimensions:

sz=size(zcm)
nlon=sz[1]
nlat=sz[2]
nlev=sz[3]

; Assign constants:

mubar=cos(60.*!pi/180.)     ; cosine of average zenith angle (60 degrees) (=0.5)
xo2= 7.0e-18                ; O2 absorpton cross section, cm-2
xox0=2.5e-18                ; O resonant scattering cross section at line center, cm-2
a1356=4.540e3               ; Transition coefficient for 135.6
a1359=1.340e3               ; Transition coefficient for 135.9
r1356=a1356/(a1356+a1359)
r1359=a1359/(a1356+a1359)

; Apply temperature dependent ground state population to cross section:

xoxt=xox0*(0.812+(tn-200)*(0.617-0.812)/(1000.-200.))*1.15

; Calculate dz and optical depth in each layer:

dz=fltarr(nlon,nlat,nlev)
for k=1,nlev-2 do dz[*,*,k]=(zcm[*,*,k+1]-zcm[*,*,k-1])/2.
dz[*,*,0]=dz[*,*,1]
dz[*,*,nlev-1]=dz[*,*,nlev-2]
tau=(o2*xo2+ox*xoxt)*dz/mubar

; single scattering of 135.6:

init=vero5s*r1356
sssf=fltarr(nlon,nlat,nlev)
for k=1,nlev-2 do begin
  for kk=0,k-1 do $
  sssf[*,*,k]=sssf[*,*,k]+xoxt[*,*,k]*ox[*,*,k]*init[*,*,kk]*dz[*,*,kk]*exp(-total(tau[*,*,kk:k],3))
  for kk=k+1,nlev-1 do $
  sssf[*,*,k]=sssf[*,*,k]+xoxt[*,*,k]*ox[*,*,k]*init[*,*,kk]*dz[*,*,kk]*exp(-total(tau[*,*,k:kk],3))
endfor
sssf=sssf*r1356

; double scattering of 135.6:

dssf=fltarr(nlon,nlat,nlev)
for k=1,nlev-2 do begin
  for kk=0,k-1 do $
  dssf[*,*,k]=dssf[*,*,k]+xoxt[*,*,k]*ox[*,*,k]*sssf[*,*,kk]*dz[*,*,kk]*exp(-total(tau[*,*,kk:k],3))
  for kk=k+1,nlev-1 do $
  dssf[*,*,k]=dssf[*,*,k]+xoxt[*,*,k]*ox[*,*,k]*sssf[*,*,kk]*dz[*,*,kk]*exp(-total(tau[*,*,k:kk],3))
endfor
dssf=dssf*r1356

; triple scattering of 135.6

tssf=fltarr(nlon,nlat,nlev)
for k=1,nlev-2 do begin
  for kk=0,k-1 do $
  tssf[*,*,k]=tssf[*,*,k]+xoxt[*,*,k]*ox[*,*,k]*dssf[*,*,kk]*dz[*,*,kk]*exp(-total(tau[*,*,kk:k],3))
  for kk=k+1,nlev-1 do $
  tssf[*,*,k]=tssf[*,*,k]+xoxt[*,*,k]*ox[*,*,k]*dssf[*,*,kk]*dz[*,*,kk]*exp(-total(tau[*,*,k:kk],3))
endfor
tssf=tssf*r1356

; Add scattered photons proportionally to 135.6 and 135.9 total final source functions:

tfsf1356 = init + sssf + dssf + tssf
tfsf1359 = vero5s*r1359 +(sssf+dssf+tssf)*r1359/r1356

return

end
