;+
; NAME:
;	ECR_TO_ECI
;
; PURPOSE:
;	Converts position vectors from ECR coordinates to ECI coordinates
;
; CATEGORY:
;	Utility
;
; CALLING SEQUENCE:
;	ECR_TO_ECI, year, doy, utc, ecr, eci
; 
; INPUTS:
;	year	Year, yyyy, longword integer
;	doy	Day of Year, ddd, longword integer
;	utc	Coordinated Universal Time of day in seconds, floating point
;	ecr	Position vector in Earth-centered-rotational coordinates, x, y, z, in km
;
; OUTPUTS:
;       eci	Position vector in Earth-centered-intertial coordiantes, x, y, x, in km.
;
; KEYWORDS:
;	None
;
; COMMON BLOCKS:
;	None.
;
; PROCEDURE:
;	Transform position vectors in ECR (Earth centered rotational) coordinates
;	into Earth-Centered-Inertial position vectors.
;	Uses SUNCOR to find Greenwich sidereal time (GST), the angle between
;	the Greenwich meridian and the vernal equinox.  
;	Arrays of vectors are OK (only 3 x N).
;	
; ROUTINES USED:
;	GST - calulates Greenwich sidereal time
;
; MODIFICATION HISTORY:
;       Stan Solomon, 3/15
;       Stan Solomon, 11/15 -  Changed names from GEO to ECR.
;
;-

pro ecr_to_eci, year, doy, utc, ecr, eci

; Get Greenwich sidereal time:
gst = gst(year,doy,utc)

; Calculate ECI position vector:
eci=double(ecr)
eci[0,*]=ecr[0,*]*cos(gst)-ecr[1,*]*sin(gst)
eci[1,*]=ecr[0,*]*sin(gst)+ecr[1,*]*cos(gst)

return

end
