;+
; NAME:
;	SZA
;
; PURPOSE:
;	Calculates solar zenith angle from lat/lon inputs
;
; CATEGORY:
;	Utility
;
; CALLING SEQUENCE:
;	result = sza(date,time,lat,lon,[szanoon])
; 
; INPUT PARAMETERS:
;	date	date in yyyyddd format; longword integer
;	time	universal time in seconds; floating point
;	lat     geographic latitude in degrees; floating point 
;       lon     longitude in degrees; floating point
;
; OUTPUT PARAMETERS:
;	sza	solar zenith angle, degrees; floating point double precision
;       szanoon optional output of solar zenith angle at local noon
;
; COMMON BLOCKS:
;	none
;
; ROUTINES USED:
;	SUNCOR - calculates coordinates of sun and Greenwich sidereal time
;
; PROCEDURE:
;	Calculates solar zenith angle for geographic coordinates
;       Arrays are OK.
;	No error checking.
;
;
; MODIFICATION HISTORY:
;       Based on SUNCOR v. 1.0 (12/01) and LLA_TO_ECI v. 2.0A (4/03)
;	6/04	Version 1.0	Stan Solomon
;-

function sza, date, time, lat, lon, szanoon

; Get solar coordinates and Greenwich sidereal time:
suncor, date, time, sdec, sra, gst

; Calculate  right ascension:
ra = atan(sin(lon*!dpi/180.+gst),cos(lon*!dpi/180.+gst))

; Calculate declination:
f = 1./298.257D   ; Earth oblateness flattening factor
dec = atan(tan(lat*!dpi/180.)*(1.-f)^2)

; Calculate solar zenith angle:
sza = acos(sin(sdec)*sin(dec)+cos(sdec)*cos(dec)*cos(ra-sra))

; Convert radians into degrees:
sza = sza * 180./!dpi

; Optional return - SZA at local noon:
szanoon = abs(dec-sdec) * 180./!dpi

return,sza

end
