;+
; NAME:
;	WRITECUBEFILE
;
; PURPOSE:
;	Writes a file containing the output from GOLDLOS
;
; CATEGORY:
;	Output utility
;
; CALLING SEQUENCE:  
;	writecubefile,ntime,outfilepath,amodel,gmodel,date,ut,f107,f107a,ap,chi,psi,wave,scb, $
;	scbo5s,scblbht,scblbhs,scblbh30,altref,latref,lonref,lstref,szaref,ezaref,tref,teff,flag
;
; INPUTS:
;	Frame number, output file path, metadata, and fields from GOLDLOS (see defs array)
;
; OUTPUTS:  
;	netCDF file containing GOLDLOS output
;
; COMMON BLOCKS:
;	None
;
; PROCEDURE:
;	Assembles file name, makes structure, writes output
;
; ROUTINES CALLED:
;	WRITE_NETCDF
;
; MODIFICATION HISTORY:
;	Stan Solomon, 2016
;
;+

pro writecubefile,ntime,outfilepath,amodel,gmodel,date,ut,f107,f107a,ap,chi,psi,wave,scb, $
    scbo5s,scblbht,scblbhs,scblbh30,altref,latref,lonref,lstref,szaref,ezaref,tref,teff,flag

      wavelength=wave/10.

      defs=strarr(25)
      defs[0] ='definitions: string array specifying variable definintions                      '
      defs[1] ='amodel: string defining source model, e.g., TIE-GCM, TIME-GCM, MSIS/IRI, WACCM-X'
      defs[2] ='gmodel: string defining version of GLOW model used                              '
      defs[3] ='date: date in yyddd format                                                      '
      defs[4] ='ut: universal time in seconds                                                   '
      defs[5] ='f107: daily F10.7 index                                                         '
      defs[6] ='f107a: F10.7 index 81-day centered average                                      '
      defs[7] ='ap: Ap index (daily if MSIS/IRI run, derived from HP or 3-hour Kp if TGCM run)  '
      defs[8] ='chi: grid angle from satellite toward Earth in east-west direction (deg., +E)   '
      defs[9] ='psi: grid angle from satellite toward Earth in north-south direction (deg., +N) '
      defs[10]='wavelength: wavelength at center of bin (nm)                                    '
      defs[11]='imagecube: array of slant column brightness [chi,psi,wavelength] (Rayleighs)    '
      defs[12]='scbo5s: slant column brightness of O(5S) doublet at 135.6 & 135.9 nm (Rayleighs) '
      defs[13]='scblbht: slant column brightness of the LBH bands 137-160 nm (Rayleighs)        '
      defs[14]='scblbhs: slant column brightness of the LBH "short" bands 141-153 nm (Rayleighs)'
      defs[15]='scblbh30: slant column brightness of the LBH (3,0) band at 135.4 nm (Rayleighs) '
      defs[16]='altref: reference altitude (km) (using 160 km for disk, tangent point for limb) '
      defs[17]='latref: latitude at reference altitude at each grid point (degrees)             '
      defs[18]='lonref: longitude at reference altitude at each grid point (degrees)            '
      defs[19]='lstref: local solar time at reference altitude at each grid point (hours)       '
      defs[20]='szaref: solar zenith angle at reference altitude each grid point (degrees)      '
      defs[21]='ezaref: emission zenith angle at reference altitude at each grid point (deg.)'
      defs[22]='tref: temperature at reference altitude at each grid point, disk only (K)       '
      defs[23]='teff: effective temperature weighted along line of sight at each grid point (K) '
      defs[24]='flag: flag indicating type of observation, 0=nothing, 1=limb, 2=disk            '
      
      struc = {definitions:defs, amodel:amodel, gmodel:gmodel, $
               date:date, ut:ut, f107:f107, f107a:f107a, ap:ap, $
               chi:chi, psi:psi, wavelength:wavelength, $
               imagecube:scb, scbo5s:scbo5s, scblbht:scblbht, scblbhs:scblbhs, scblbh30:scblbh30, $
               altref:altref, latref:latref, lonref:lonref, lstref:lstref, $
               szaref:szaref, ezaref:ezaref, tref:tref, teff:teff, flag:flag}

      outfile=outfilepath+'.'+string(ntime,format='(i3.3)')+'.nc'
      write_netcdf,struc,outfile,att_file='attributes.txt'

    return

end
